# ----------------------------------------------------------------------
#  SVD-Control main file
# ----------------------------------------------------------------------
# 
package require Iwidgets

#
# this variable allows easy access to files that are either in
# a bundled TCL or in the "current directory".  If this script
# is running bundled, the [info script] comes back with the
# interal archive pathname.  Otherwise, the simple pathname to
# the running script is returned.

set home [file dirname [info script]]

#
# the following is a hack to get the name of the directory in
# which the executable programs are running.  In windows, the
# directory is the "current" directory, but in Linux it is
# not necessarily.
#
# Note that this hack DEPENDS upon the fact that this particular
# part of the application is name SVD-Control.tcl  .  It wasn't
# completely necessary that this is the case, however.

if { ! [regexp {^(.*)/[^/]*/lib/application/SVD-Control.tcl} [info script] ignorematch basepath] } {
    set basepath "."
}


#
# This code is the "initial" start-up code that puts up a
# window right away, then loads the rest of the bulk of this
# thing.
#
source [file join $home ini.tcl]
source [file join $home message.tcl]

#
# change directory to the installed directory, and
# make sure that the current directory is in the PATH
#
cd $basepath
set env(PATH) ".:$env(PATH)"

setIniDefaults
iniRead

wm geometry . [iniGetAttribute "Program" "Position"]

workingMessageCreate
workingMessagePopUp "Loading...please wait..."

source [file join $home serial.tcl]
source [file join $home serialgui.tcl]
source [file join $home download.tcl]
source [file join $home svd.tcl]
source [file join $home extern.tcl]
source [file join $home popup.tcl]
source [file join $home oslist.tcl]
source [file join $home upload.tcl]
source [file join $home files.tcl]
source [file join $home save.tcl]
source [file join $home options.tcl]
source [file join $home tips.tcl]
source [file join $home load.tcl]

proc guiVersion {} {
    return "version 2.2 "
}

proc showVersions {} {
    return [format "%s: %s\n%s\n%s\n" "SVD-Control" [guiVersion] [tosvdVersion] [fstoolVersion]]
}

########################################################################
# gaugeCreate() -	Creates a simple progress gauge.  The value
#                       should be an integer percentage.
########################################################################
proc gaugeCreate {win color width height} {

    frame $win -borderwidth 2 -relief sunken
    canvas $win.display -borderwidth 0 -background white  \
              -highlightthickness 0 -width $width -height $height

    pack $win.display -expand yes
    $win.display create rectangle 0 0 $width $height \
        -outline "black" \
        -fill $color  \
        -tags bar

    return $win
}

########################################################################
# gaugeValue() -	The value of a gauge should be an int percentage.
########################################################################
proc gaugeValue {win val recolor} {
    if {$val < 0 } {
        error "bad value \"$val\": should be 0-100"
    }

    if {$val == 0} {
	$win.display itemconfigure bar -outline ""
	$win.display itemconfigure bar -fill "white"
    } else {
	$win.display itemconfigure bar -outline "black"
    }

    $win.display itemconfigure bar -fill "green"

    if {$val > 80 && $recolor } {
	$win.display itemconfigure bar -fill "yellow"
    }

    if {$val > 99} {
	set val 100
	if { $recolor } {
	    $win.display itemconfigure bar -fill "red"
	}
    }

    set w [expr 0.01 * $val * ([winfo width $win.display]-2)]
    set h [expr [winfo height $win.display] - 2]

    $win.display coords bar 1 1 $w $h
}

#################################################
### Paint the main menu bar
#################################################

proc menuCreate {} {
  iwidgets::menubar .mb -helpvariable helpVar -menubuttons {
    menubutton file -text "File" -menu {
        options -tearoff false 

	cascade ftype -label "Floppy Type..." -menu {
        }
	
	command rs232 -label "RS232 Settings..." -underline 0 \
		-helpstr "Set RS232 device and baud" \
		-command { floppyRS232popup }

	command options -label "Options..." -underline 0 \
		-helpstr "Set program options" \
		-command { optionsDialogActivate }

        separator sep2

        command save -state normal -label "Save" -command {iniWrite} \
                -helpstr "Save current configuration"

        command exit -state disabled -label "Exit" -command { saveExit } \
                -helpstr "Exit application (configuration will be saved)"
    }

    menubutton disk0 -text "Disk 0" -menu {
        options -tearoff false

        command load -label "Load..." -underline 0 \
                -helpstr "Load or change floppy" \
                -command { floppyLoad 0 }

        command eject -label "Eject" -underline 0 \
                -helpstr "Eject (clear) this floppy" \
                -command { floppyEject 0 }

        separator sep2

        checkbutton enable -label "Enabled" -underline 1 \
                -helpstr "Enable/disable this floppy" \
		-variable floppyEnabled(0) \
                -command {
	            menuFloppyEnableToggle 0
		}

        checkbutton protect -label "Write Protect" -underline 0 \
                -helpstr "Write protect this floppy" -variable floppyProtected(0)

        separator sep3

	command swap1 -label "Swap with disk 1" -underline 15 \
		-helpstr "Swap disk 0 with disk 1" -command { floppySwap 0 1 }
	command swap2 -label "Swap with disk 2" -underline 15 \
		-helpstr "Swap disk 0 with disk 2" -command { floppySwap 0 2 }

	separator sep4

	command upload -label "Upload from SVD..." -underline 0 \
		-helpstr "Upload disk 0 from SVD" -command { floppyUpload 0 }

	command save -label "Save Image..." -underline 0 \
		-helpstr "Save disk 0 image to a new file" -command { floppySave 0 }
    }

    menubutton disk1 -text "Disk 1" -menu {
        options -tearoff false

        command load -label "Load..." -underline 0 \
                -helpstr "Load or change floppy" \
                -command { floppyLoad 1 }

        command eject -label "Eject" -underline 0 \
                -helpstr "Eject (clear) this floppy" \
                -command { floppyEject 1 }

        separator sep2

        checkbutton enable -label "Enabled" -underline 1 \
                -helpstr "Enable/disable this floppy" -variable floppyEnabled(1) \
                -command {
	            menuFloppyEnableToggle 1
		}

        checkbutton protect -label "Write Protect" -underline 0 \
                -helpstr "Write protect this floppy" -variable floppyProtected(1)

        separator sep3

	command swap1 -label "Swap with disk 0" -underline 15 \
		-helpstr "Swap disk 1 with disk 0" -command { floppySwap 1 0 }
	command swap2 -label "Swap with disk 2" -underline 15 \
		-helpstr "Swap disk 1 with disk 2" -command { floppySwap 1 2 }

	separator sep4

	command upload -label "Upload from SVD..." -underline 0 \
		-helpstr "Upload disk 1 from SVD" -command { floppyUpload 1 }

	command save -label "Save Image..." -underline 0 \
		-helpstr "Save disk 1 image to a new file" -command { floppySave 1 }
    }

    menubutton disk2 -text "Disk 2" -menu {
        options -tearoff false

        command load -label "Load..." -underline 0 \
                -helpstr "Load or change floppy" \
                -command { floppyLoad 2 }

        command eject -label "Eject" -underline 0 \
                -helpstr "Eject (clear) this floppy" \
                -command { floppyEject 2 }

        separator sep2

        checkbutton enable -label "Enabled" -underline 1 \
                -helpstr "Enable/disable this floppy" -variable floppyEnabled(2) \
                -command {
	            menuFloppyEnableToggle 2
		}

        checkbutton protect -label "Write Protect" -underline 0 \
                -helpstr "Write protect this floppy" -variable floppyProtected(2)

        separator sep3

	command swap1 -label "Swap with disk 0" -underline 15 \
		-helpstr "Swap disk 2 with disk 0" -command { floppySwap 2 0 }
	command swap2 -label "Swap with disk 1" -underline 15 \
		-helpstr "Swap disk 2 with disk 1" -command { floppySwap 2 1 }

	separator sep4

	command upload -label "Upload from SVD..." -underline 0 \
		-helpstr "Upload disk 2 from SVD" -command { floppyUpload 2 }

	command save -label "Save Image..." -underline 0 \
		-helpstr "Save disk 2 image to a new file" -command { floppySave 2 }
    }

    menubutton options -text "SVD" -menu {
        options -tearoff false -selectcolor blue

        command reset -label "Reset" -underline 0 \
                -helpstr "Reset the SVD (unloads all disks)" \
                -command {
	            tipsDialogShow "Reset"
	            svdReset
	}

        command check -label "Check" -underline 0 \
                -helpstr "Checks communication and version of SVD" \
                -command { svdCheck 1 }

        command updateall -label "Download" -underline 0 \
                -helpstr "Download all floppy images to SVD" \
                -command { downloadActivate }

        separator sep

        checkbutton auto -label "Auto Download" -underline 0 \
                -helpstr "Automatically download floppies" \
                -variable svdAutoLoad \
		-command { autoLoadSet }
    }
    menubutton help -text "Help" -menu {
	options -tearoff false -selectcolor blue

	command tipreset -label "Turn on All Tips..." -underline 0 \
		-helpstr "Turns on all SVD Helpful Tips (show them again)" \
		-command { tipsReset }

	command tip -label "Show Tip" -underline 5 \
		-helpstr "Show a helpful SVD tip" \
		-command { tipsDialogShow "General" }

	command search -label "Show All Tips..." -underline 0 \
		-helpstr "Show all of the program tips (and allow searching)" \
		-command { .tipshow activate }

        separator sep

	command web -label "Where to find more help..." -underline 0 \
		-helpstr "Where can I find help with this program?" \
		-command { 
	             tk_messageBox -default ok -icon info -title "Where to find Help"  \
			-type ok -message "Unfortunately, help isn't built into this program.  Sorry about that.  However, you can find a bunch of help at the SVD web-site:  \n\n\thttp://www.theSVD.com"
	         }

	command version -label "Version..." -underline 0 \
		-helpstr "Show the version numbers of this program" \
		-command { 
	             tk_messageBox -default ok -icon info -title "Version Info" \
			     -type ok -message {[showVersions]}
	         }
    }
  }
}


proc menuFloppyEnableToggle {num} {
    global floppyEnabled
    global floppyMemoryUsed

    if { $floppyEnabled($num) } {
	svdMemoryUsageIncrRaw $floppyMemoryUsed($num)
    } else {
	svdMemoryUsageDecr $floppyMemoryUsed($num)
    }
}

###############################
# pieces of the canvas
###############################

frame .disks
# image create photo diskimage -file [file join $home trsdisk2.gif]
# image create photo diskimage -file "generic.gif"

############
# DISKS
############

proc diskCluster {num} {
    iwidgets::Labeledframe .disks.disk($num) -labelpos nw -labeltext "Disk $num"
    set area [.disks.disk($num) childsite]
     
    checkbutton $area.wp -highlightthickness 0 -anchor w -justify left \
	    -text "Write Protect" -underline 2 -variable floppyProtected($num)

    checkbutton $area.enabled -highlightthickness 0 -anchor w -justify left \
	    -text "Enabled" -underline 0 -variable floppyEnabled($num) \
	    -command [list menuFloppyEnableToggle $num]

    label $area.picture -image diskimage

    label $area.file -text "" -relief flat

    frame $area.buttons

    eval button $area.buttons.load -text "Load..." -command "{ floppyLoad $num }"
    eval button $area.buttons.eject -text "Eject" -command "{ floppyEject $num }"

    pack $area.buttons.load -side left -expand yes -fill x -padx 1 -pady 1
    pack $area.buttons.eject -side left -expand yes -fill x -padx 1 -pady 1

    gaugeCreate $area.gauge white 80 10

    pack $area.picture -side top
    pack $area.file -side top
    pack $area.enabled -side top -fill x -anchor w
    pack $area.wp -side top -fill x -anchor w
    pack $area.buttons -side top -fill x -anchor w -expand yes
    pack $area.gauge -side bottom -expand yes -padx 1 -pady 2
}

########################
# SVD Control frame
########################

proc svdFrameCreate {} {
    iwidgets::Labeledframe .svd -labelpos nw -labeltext "SVD Control"
    set svd [.svd childsite]

    frame $svd.info
    label $svd.info.vlabel -text "SVD Version"
    label $svd.info.version -text "" -relief sunken
    label $svd.info.blank -text ""
    pack $svd.info.vlabel -side top
    pack $svd.info.version -side top -expand yes -fill x
    pack $svd.info.blank -side top

    frame $svd.gauge
    label $svd.gauge.title -text "Memory Usage"
    gaugeCreate $svd.gauge.bar white 90 10
    label $svd.gauge.percent -text "0%"
    pack $svd.gauge.title -side top
    pack $svd.gauge.bar -side top -pady 2
    pack $svd.gauge.percent -side top

    frame $svd.update
    checkbutton $svd.update.auto -highlightthickness 0 -anchor w -justify left \
        -text "Auto Download" -underline 0 -variable svdAutoLoad \
	-command { autoLoadSet }
    button $svd.update.all -text "Download All" \
	-command { downloadActivate }
    pack $svd.update.auto -side top -fill x -expand yes
    pack $svd.update.all -side top -fill x -expand yes

    frame $svd.control
    button $svd.control.reset -text "Reset" -command { 
	tipsDialogShow "Reset"
	svdReset
    }
    button $svd.control.check -text "Check" -command { svdCheck 1 }
    pack $svd.control.reset -padx 2 -pady 2 -side top -anchor n -expand yes
    pack $svd.control.check -padx 2 -pady 2 -side top -anchor n -expand yes

    pack $svd.info -side left -expand yes
    pack $svd.gauge -side left -padx 4 -pady 4 -expand yes

    pack $svd.update -padx 4 -pady 4 -side left -anchor n -expand yes
    pack $svd.control -padx 4 -pady 4 -side left -anchor n -expand yes
}
 
########################################################################
# svdUpdateVersion() -	Update the version of the SVD on the display.
########################################################################

proc svdUpdateVersion {} {
    global svdMajorVersion
    global svdMinorVersion
    
    if { $svdMajorVersion == 1 && $svdMinorVersion == 5 } {

	tk_messageBox -default ok -icon warning -title "Version Warning" \
		-type ok -message "This version of the SVD Control Program is not \
compatible with version 1.5 of the hardware. \
This operation will most likely fail. \
\
Please see http://www.theSVD.com for upgrade \
information."

	[.svd childsite].info.version configure -text [format "*SVD %d.%d*" $svdMajorVersion $svdMinorVersion]

    } elseif { $svdMajorVersion != 2 } {

	tk_messageBox -default ok -icon warning -title "Version Warning" \
		-type ok -message "This version of the SVD Control Program is not \
compatible with your version of the hardware. \
This operation will most likely fail."

	[.svd childsite].info.version configure -text [format "*SVD %d.%d*" $svdMajorVersion $svdMinorVersion]

    } else {

	[.svd childsite].info.version configure -text [format "SVD %d.%d" $svdMajorVersion $svdMinorVersion]

    }
}

#########################################
# Paint the basic canvas
#########################################
proc paintScreen {} {
    menuCreate

    diskCluster 0
    diskCluster 1
    diskCluster 2

    svdFrameCreate

    pack .mb -fill x -side top

    pack .disks.disk(0) -side left
    pack .disks.disk(1) -side left
    pack .disks.disk(2) -side left

    pack .disks -fill x -side top

    label .help -anchor w -textvariable helpVar -relief raised
    pack .help -anchor w -side bottom -expand yes -fill x

    pack .svd -anchor w -expand yes -fill both -side bottom
}

########################################################################
# gaugeSetPercent() -	Set the given percentage on the given super-gauge
########################################################################
proc gaugeSetPercent {win percent} {
    gaugeValue $win.bar $percent 1
    $win.percent configure -text [format "%d %%" $percent]
}

set svdMemoryBlocks 1024
set svdMemoryUsed 0
set svdDDSupported 0

########################################################################
# svdVersionSet() -	Sets internal variables that are specific to the
#                       different versions of the SVD firmware.
#
# version 1.2 of the svd has 256k of memory, or 1024 blocks, and only SD
# version 1.5 has the same memory config, but also has DD
#
########################################################################
proc svdVersionSet {} {
    global svdMajorVersion
    global svdMinorVersion

    global   svdMemoryBlocks
    global   svdDDSupported

    set svdMemoryBlocks 1024

    if { $svdMajorVersion < 2 && $svdMinorVersion < 5 } {
	set svdDDSupported 0 
    } else {
	set svdDDSupported 1
    }
}
    
########################################################################
# svdMemoryUsageCalc() -	Calculates the memory used by the given
#                               floppy parameters.
#                               of the amount of memory in use.  NOTE -
#                               use negative numbers for decrement.
# Note that "totalslop" below is the worst case slop factor.
########################################################################
proc svdMemoryUsageCalc {sides tracks sectors} {
    global svdMemoryUsed
    global svdMemoryBlocks

    set incr [expr $sides * $tracks * ($sectors+1)]
    set slop [expr 256 % ($sectors+1)]
    set totalslop [expr ($sides * $tracks * ($sectors+1) / 256 + 1) * $slop]

    return [expr $incr + $totalslop]
}

proc svdMemoryUsageIncrRaw {amount} {
    global svdMemoryUsed
    global svdMemoryBlocks

    set svdMemoryUsed [expr $svdMemoryUsed + $amount]

    gaugeSetPercent [.svd childsite].gauge [expr $svdMemoryUsed * 100 / $svdMemoryBlocks]
}

proc svdMemoryUsageDecr {amount} {
    global svdMemoryUsed
    global svdMemoryBlocks

    set svdMemoryUsed [expr $svdMemoryUsed - $amount]

    gaugeSetPercent [.svd childsite].gauge [expr $svdMemoryUsed * 100 / $svdMemoryBlocks]
}

proc floppyMemoryIndicator {num} {
    global floppyMemoryUsed
    global svdMemoryBlocks

    gaugeValue [.disks.disk($num) childsite].gauge \
	    [expr $floppyMemoryUsed($num) * 100 / $svdMemoryBlocks] 1
}

###########################################
# procs for enabling/disabling floppy's GUI
###########################################

proc floppyGuiDisable {num } {

    set area [.disks.disk($num) childsite]

    .mb menuconfigure .disk$num.load -state normal
    .mb menuconfigure .disk$num.eject -state disabled
    .mb menuconfigure .disk$num.enable -state disabled
    .mb menuconfigure .disk$num.protect -state disabled
    .mb menuconfigure .disk$num.swap1 -state disabled
    .mb menuconfigure .disk$num.swap2 -state disabled
    .mb menuconfigure .disk$num.upload -state normal
    .mb menuconfigure .disk$num.save -state disabled

    $area.picture configure -state disabled
    $area.buttons.load configure -state normal
    $area.buttons.eject configure -state disabled
    $area.enabled configure -state disabled
    $area.wp configure -state disabled

    floppyPopUpDisable $num

    $area.file configure -relief flat
    $area.file configure -text ""

# have a problem on Linux - the buttons stay in "focused" color
# these events seem to help - I think this was a problem with
#  using "active" instead of normal...see if it works without them
#
#    event generate $area.buttons.eject <Leave>
#    event generate $area.buttons.load <Leave>
#    event generate $area.update <Leave>
#    event generate $area.enabled <Leave>
#    event generate $area.wp <Leave>
#    event generate $area.picture <Leave>
}

proc floppyGuiEnable {num } {
    global floppyType
    global floppyImage

    set area [.disks.disk($num) childsite]

    .mb menuconfigure .disk$num.load -state normal
    .mb menuconfigure .disk$num.eject -state normal
    .mb menuconfigure .disk$num.enable -state normal
    .mb menuconfigure .disk$num.protect -state normal
    .mb menuconfigure .disk$num.swap1 -state normal
    .mb menuconfigure .disk$num.swap2 -state normal
    .mb menuconfigure .disk$num.upload -state normal
    .mb menuconfigure .disk$num.save -state normal

    $area.picture configure -state normal
    $area.buttons.load configure -state normal
    $area.buttons.eject configure -state normal
    $area.enabled configure -state normal
    $area.wp configure -state normal

    floppyPopUpEnable $num

    $area.file configure -relief sunken
    if {$floppyType($num) == "files"} {
	$area.file configure -text "(mixed files)"
    } elseif {$floppyType($num) == "upload"} {
	$area.file configure -text "(upload)"
    } else {
	set base ""
	regexp {.*/([^/]*)} $floppyImage($num) match base
	$area.file configure -text $base
    }

# have a problem on Linux - the buttons stay in "focused" color
# these events seem to help - I think this was a problem with
#  using "active" instead of normal...see if it works without them
#
#    event generate $area.buttons.eject <Leave>
#    event generate $area.buttons.load <Leave>
#    event generate $area.update <Leave>
#    event generate $area.enabled <Leave>
#    event generate $area.wp <Leave>
#    event generate $area.picture <Leave>
}


#######################################################
# initialization for GUI - includes menu and buttons
#######################################################
proc guiInit {} {
    paintScreen
    .mb menuconfigure .file.exit -state normal

    floppyGuiDisable 0
    floppyGuiDisable 1
    floppyGuiDisable 2
}

proc floppyEject { num } {
    global floppyEnabled
    global floppyProtected
    global floppyLoaded
    global floppyMemoryUsed
    global floppyType
    global floppyFiles
    global floppyFilesOS
    global floppyImage
    global floppyUploadFile
    global OSListSimple

    if { $floppyLoaded($num) } {
	if { $floppyType($num) == "upload" } {
	    set answer [tk_messageBox -default yes -icon warning -title "WARNING"  \
		                      -type yesnocancel \
				      -message \
"Floppy $num has an uploaded image that hasn't been saved. \
Do you want to save it before it is lost?"]

            if { $answer == "yes" } {
		if { ! [floppySave $num] } {
		    return 0
		}
	    } elseif { $answer == "no" } {
		file delete -force $floppyUploadFile($num)
	    } else {
		return 0
	    }
	    set floppyUploadFile($num) ""
	}
	floppyGuiDisable $num
	if { $floppyEnabled($num) } {
	    svdMemoryUsageDecr $floppyMemoryUsed($num)
	}
    }

    set floppyEnabled($num) 0
    set floppyProtected($num) 0
    set floppyLoaded($num) 0

    set floppyType($num) "image"
    set floppyFiles($num) ""
    set floppyImage($num) ""
    set floppyFilesOS($num) [lindex "$OSListSimple" 0]

    set floppyMemoryUsed($num) 0

    floppyMemoryIndicator $num

    return 1
}

proc autoLoadSet {} {
    global svdAutoLoad

    iniSetAttribute "Program" "AutoLoad" $svdAutoLoad
}

#  floppyCurrent has the floppy we're working on
#  floppyType has the type of the floppy
#  need to have an array of the floppies that are loaded
#    and the data that is in the SVD which can be compared
#  Also, when working on floppy data, we'll use a temporary
#    variable so that cancel can be pressed at any time.
#  When coming back it is loaded into the right floppy.

set floppyCurrent 0

########################################################################
# floppyLoad() -	Called when the load button is pressed
########################################################################

proc floppyLoad { num } {
    tipsDialogShow "Load"
    floppyFilesDialogActivate $num
}

#####################################
# floppy data
#   number - 0 to 2
#   floppyLoaded() - boolean - has it been loaded
#   floppyType() - image or files
#   floppyImage() - single file
#   floppyUploadFile() - uploaded file
#   floppyFiles() - list of files (file & date)
#   floppyEnabled() - boolean - is it enabled
#   floppyProtected() - boolean - is it write-protected
#####################################

#####################################
# SVD data - data concerning what has been
#    loaded into the SVD - used for update
#  Exact copy of the floppy data that has
#   been loaded.  When the floppy data is
#   out of sync, then a reload is necessary
#####################################

# Note that the tosvd -m format is as follows:
#
# *		 format,wp,sides,tracks,sectors,sectorexcep,density,densityexcep
# *
# *		such as:
# *
# *		DMK,1,1,40,11,1,0,1
# *
# *	Which means DMK format, write-protected, 1 side, 40 tracks, 11 sectors
# *		with some "exceptions", single density with some exceptions
#
# Version 1.2 of the SVD firmware couldn't handle either DD or weird mixes of
# the number of sectors and/or density.  v1.5 and beyond handle it just fine.
# The error messages below are mostly used for v1.2.

set bad_format_message "The image file you specified isn't readable or is in a format I can't handle.  I refuse to go on."
set density_warning "The image file you specified is double density.\nYour version of the SVD cannot handle that.\nPlease upgrade today!"
set exceptions_warning "The image file you specified has some funky mixed sectors.\nYour version of the SVD cannot handle that.\nPlease upgrade today!"

########################################################################
# loadFloppyTypes() -	Find out what floppy types are supported.
#                       This simply supplied default data for disk
#                       images and extensions.
########################################################################
proc loadFloppyTypes {} {
    floppyTypeOSList
}

proc loadFloppyTypesMenu {} {
    global floppyTypes

    .mb add radiobutton .file.ftype.generic -label "Generic Floppy" \
	    -helpstr "Generic Floppy - no file system support" \
	    -variable FloppyType -value generic -command { floppyTypeSwitch }

    set searchid [array startsearch floppyTypes]

    while { [set n [array nextelement floppyTypes $searchid]] != "" } {

	.mb add radiobutton .file.ftype.$n -label $floppyTypes($n) \
		-helpstr "$floppyTypes($n) ($n is the file system type)" \
		-variable FloppyType -value $n -command { floppyTypeSwitch }

    }

    array donesearch floppyTypes $searchid
}

########################################################################
# floppyClear() -	Clears all of the data for the displayed floppies
########################################################################
proc floppyClear {} {
    for {set i 0} { $i < 3 } {incr i} {
	floppyEject $i
    }
}

########################################################################
# floppyVarInit() -	Initialize all of the floppy variables.
########################################################################
proc floppyVarInit {} {
    global floppyLoaded
    global floppyType
    global floppyFiles
    global floppyImage
    global floppyUploadFile
    global floppyEnabled
    global floppyProtected
    global floppyFilesOS
    global floppyMemoryUsed

    for {set i 0} { $i < 3 } {incr i} {
	set floppyLoaded($i) 0
	set floppyType($i) "image"
	set floppyFiles($i) ""
	set floppyImage($i) ""
	set floppyUploadFile($i) ""
	set floppyEnabled($i) 0
	set floppyProtected($i) 0
	set floppyFilesOS($i) ""
	set floppyMemoryUsed($i) 0
    }

}

########################################################################
# floppyTypeSwitch() -	Switches to the floppy type given in FloppyType
########################################################################
proc floppyTypeSwitch {} {
    global FloppyType

    loadOSList $FloppyType
    floppyFilesDialogLoadOS
    floppyClear
    loadFloppyPicture $FloppyType

    if { $FloppyType == "generic" } {
	iniDelAttribute "Floppy" "Type"
    } else {
	iniSetAttribute "Floppy" "Type" $FloppyType
    }
}

########################################################################
# loadFloppyPicture() -	Sets up the display of the pretty picture of
#                       the floppy, given the floppyType that is in use.
#                       If the type given is either nil, or a suitable
#                       picture isn't available, then a default picture
#                       is used.
########################################################################
proc loadFloppyPicture {ftype} {
    global OSListFile
    global home

    set genericgif [file join $home "generic.gif"]

    if { $ftype == "generic" || [array names OSListFile -exact "icon"] == "" } {
	set targetgif $genericgif
    } else {
	set targetgif $OSListFile(icon)
    }

    if { [file readable $targetgif] == 0 } {
	set targetgif $genericgif
    }

    image create photo diskimage -file $targetgif
}

########################################################################
# checkFloppyType() -	Checks to make sure that the given floppy type
#                       is supported according to the types loaded.
#                       Return TRUE if it is, FALSE otherwise
########################################################################
proc checkFloppyType {type} {
    global floppyTypes

    if { $type == "generic" } {
	return TRUE
    }
	
    set searchid [array startsearch floppyTypes]

    while { [set n [array nextelement floppyTypes $searchid]] != "" } {

	if { $type == $n } {
	    array donesearch floppyTypes $searchid
	    return TRUE
	}
    }

    array donesearch floppyTypes $searchid
    return FALSE
}

########################################################################
# saveExit() -	Save the current attributes and exit.
########################################################################
proc saveExit {} {
    global floppyLoaded

    for {set i 0} { $i < 3 } {incr i} {
	if { $floppyLoaded($i) } {
	    if { ! [floppyEject $i] } {
		return
	    }
	}
    }
		
    regexp {[0-9]+x[0-9]+(\+[0-9]+\+[0-9]+)} [wm geometry .] ignore position
    iniSetAttribute "Program" "Position" $position
    iniWrite
    exit
}

proc init {} {
    global FloppyType
    global svdAutoLoad
    global downloadStopVar
    global floppyLoaded
    global floppyType
    global floppyTypes
    global floppyFiles
    global floppyImage
    global floppyUploadFile
    global floppyEnabled
    global floppyProtected
    global floppyFilesOS
    global floppyMemoryUsed
    global OSListSimple
    global FloppyDescriptions
    global genericExts
    global supportedOSTypes


#
# set default fonts right up front so that start-up error
# messages and such may be otherwise caught
#
    option add *Dialog.msg.font [iniGetAttribute "Font" "Message"] startupFile

    # load the types of floppies supported.  Then check to make sure
    # that any .ini that was given is still supported.  If not, just
    # delete the .ini value, allowing it to return to the default.

    floppyVarInit

    loadFloppyTypes

    if { [array size FloppyDescriptions] == 0 } {
	set answer [tk_messageBox -default ok -icon error -title "ERROR"  \
		                 -type ok -message \
"The OSFiles directory isn't correctly loaded.
This software wasn't installed properly.  I
must exit!"]
	exit 0
    }

    foreach i [array names FloppyDescriptions] {
	set floppyTypes($i) $FloppyDescriptions($i)
    }

    if { ! [checkFloppyType [iniGetAttribute "Floppy" "Type"]] } {
	iniDelAttribute "Floppy" "Type"
    }

    set FloppyType  [iniGetAttribute "Floppy" "Type"]

    loadOSList $FloppyType
    loadFloppyPicture $FloppyType

    set genericExts [concat [floppyExts] ".svd"]
    set supportedOSTypes [fstoolFloppyTypes]

    set svdAutoLoad [iniGetAttribute "Program" "AutoLoad"]

    set downloadStopVar 1

    floppyFilesDialogCreate
    floppySaveDialogCreate
    floppyUploadDialogCreate
    optionsDialogCreate

    floppyPopUpMenusCreate

    guiInit

    floppyTypeSwitch

    floppyPopUpMenusBind

    downloadCreate
    workingMessageRemove

    loadFloppyTypesMenu

    wm protocol . WM_DELETE_WINDOW saveExit

    serialInit

    workingMessagePopUp "Probing serial ports..."
    probeSerialPorts
    workingMessageRemove

    floppyRS232settings

    floppyClear

    # in case the SVD isn't available, go ahead and set up default
    # memory blocks and DD behavior...makes a users' life easier

    global   svdMemoryBlocks
    global   svdDDSupported

    set svdMemoryBlocks 1024
    set svdDDSupported 1

    if { ! [catch {serialOpen} ] } {
#	svdConnect
    }

    tipsShowDialogCreate
    tipsSearchDialogCreate
    tipsDialogCreate
    if { ! [tipsDialogShow Welcome] } {
	tipsDialogShow Startup
    }
}

    if { $tcl_platform(platform) == "windows" } {
	wm iconbitmap . -default [file join $home floppy.ico]
    } else {
	#    set iconimage [image create bitmap -file ]
	#    wm iconbitmap . "@/usr/include/X11/bitmaps/wingdogs"
	#    wm iconbitmap . "@iconpc04.xpm"
#	wm iconbitmap . @/usr/include/X11/bitmaps/wingdogs
	wm iconbitmap . @[file join $basepath floppy4.xbm]
    }

init
